<?php


/**
 * GetAmazonAPI class 
 *
 * Product Advertising API (Amazon API)からデータを取得し、連想配列にして返します。
 * 実行条件は、PHPのバージョンが5.3.0以上、かつ、cURLをサポート。
 *
 * @author    sei2 
 * @copyright 2018 D&D Company
 * @license   https://www.gnu.org/licenses/gpl-3.0-standalone.html GNU General Public License
 * @link      http://blog1.dd-company.com/
 */


/**
 * LICENSE:
 * Copyright (C) 2018  D&D Company
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * -- Change Log --
 * 1.0.1 バグ修正, XMLパース時エラー対策(エラーハンドリング追加, 制御文字除去)
 * 1.0.0 新規作成
 */

class GetAmazonAPI{
	
	private $debug_mode = false;
	private $cacheID;
	private $request_url;
	private $curl_http_error_code;
	private $check_cache_lite = true;
	private $cache_options = array(
		'cacheDir'					=> 'cache/',// キャッシュが保存される場所
		'lifeTime'					=> 86400,	// キャッシュ有効時間
		'automaticCleaningFactor'	=> '20',	//(1/nの確率で)キャッシュファイルの自動削除
			);
	public $response;
	
	
	function __construct($params){
		$this->check_debug_mode();
		$this->check_PHP_version();
		$this->check_cache_lite();
		$this->make_cacheID($params);
		$this->make_amazonAPI_url($params);
		$this->get_api();
		if($this->debug_mode){
			$this->display_API_error();
		}
	}
	
	
	//キャッシュID作成 (timestamp,signature追加前のパラメータ使用)
	private function make_cacheID($params){
		$pairs = $this->key_sort($params);
		$cacheID = join("&", $pairs);
		$this->cacheID = $cacheID;
	}
	
	
	//amazonAPIのリクエストURL作成
	private function make_amazonAPI_url($params){
		// Set current timestamp if not set
		if (!isset($params["Timestamp"])) {
			$params["Timestamp"] = gmdate('Y-m-d\TH:i:s\Z');
		}
		
		// パラメータをキーで並べ替え、及び、urlエンコード
		$pairs = $this->key_sort($params);
		// Generate the canonical query
		$canonical_query_string = join("&", $pairs);
		// Generate the string to be signed
		$string_to_sign = "GET\n" . END_POINT1 . "\n" . END_POINT2 . "\n" . $canonical_query_string;
		// Generate the signature required by the Product Advertising API
		$signature = base64_encode(hash_hmac("sha256", $string_to_sign, SECRET_KEY, true));
		// Generate the signed URL //php5.3.0からrawurlencode()はRFC3986に対応
		$request_url = END_POINT0 . END_POINT1 . END_POINT2 . '?'.$canonical_query_string.'&Signature='.rawurlencode($signature);
		$this->request_url = $request_url;
	}
	
	
	// パラメータをキーで並べ替え、及び、urlエンコード //php5.3.0からrawurlencode()はRFC3986に対応
	private function key_sort($params){
		ksort($params);
		$pairs = array();
		
		foreach ($params as $key => $value) {
			array_push($pairs, rawurlencode($key)."=".rawurlencode($value));
		}
		
		return($pairs);
	}
	
	
	//外部データをAPI経由で取得する（キャッシュを利用）
	private function get_api(){
		
		$request_url = $this->request_url;
		$cacheID = $this->cacheID;
		$response = '';

		//Cache_Liteが有効でキャッシュモードが無効でないとき
		if($this->check_cache_lite) {
			require_once("Cache/Lite.php");
			//インスタンス生成
			$objCache = new Cache_Lite($this->cache_options);
			//有効なキャッシュが存在するとき
			if ($cache = $objCache->get($cacheID)) {
				$response = $cache;
				if($this->debug_mode){
					print('キャッシュ使用' . "\n");
				}
			}
		}
		
		// キャッシュがなかったとき、lifeTimeより古くなっているとき
		if(!$this->check_cache_lite || !$response) {
			
			//一定(最小2秒)間隔でAPIにリクエストするための処理
			sleep(2);
			
			//データ取得時、500、503エラーの場合、最大5回取得しに行く
			for($cnt = 0; $cnt< 5; $cnt++){
				$response = $this->file_get_by_curl($request_url);
				//データ取得成功(httpステータスコードが200)時はループから抜ける
				if($this->curl_http_error_code === 200){
					break;
				}
				//連続取得を1秒間隔にする
				sleep(1);
			}
			
			//5回連続データ取得失敗した時、プログラム終了する
			if($this->curl_http_error_code !== 200){
				if($this->debug_mode){
					echo 'サーバーエラー　httpステータスコード : ' . $this->curl_http_error_code . "\n";
					echo 'リクエストURL : ' . $this->request_url . "\n";
					$this->display_API_error();
				}
				exit;
			}
			
			//Cache_Liteが有効でキャッシュモードが無効でないとき、返されたデータをキャッシュに保存
			if ($this->check_cache_lite) {
				$objCache->save($response, $cacheID);
			}
		}
		
		//xmlの配列化
		$res_array = $this->xml_to_array($response);
		
		$this->response = $res_array;
	}
	
	
	//xmlを配列化
	private function xml_to_array($response){
		//ASCII制御文字除去
		$search = array(
		  "\0", "\x01", "\x02", "\x03", "\x04", "\x05",
		  "\x06", "\x07", "\x08", "\x0b", "\x0c", "\x0e", "\x0f"
		);
		$response = str_replace($search, '', $response); // ASCII制御文字を取り除く
		$response = mb_convert_encoding($response, "UTF-8", "auto"); // UTF-8に強制エンコーディング
		//simplexml_load_string() エラーの、ユーザーによるエラー処理を有効にする
		libxml_use_internal_errors(true);
		//xmlのオブジェクト化
		$res_object = simplexml_load_string($response);
		//エラー処理
		if(!$res_object){
			if($this->debug_mode){
				echo '<pre>' . "\n";
				echo '-- XML パースエラー --' . "\n";
				foreach(libxml_get_errors() as $error) {
					echo 'エラーメッセージ' . "\t: " .  $error->message;
					echo 'エラーレベル' . "\t\t: " . $error->level . "\n";
					echo 'エラーコード' . "\t\t: " . $error->code . "\n";
					echo 'エラー位置' .  "\t\t: " .  $error->line . '行 ' . $error->column . '列' . "\n";
				}
				echo '</pre>';
			}
			libxml_clear_errors();
			exit();
		}
		//オブジェクトの配列化
		$res_json = json_encode($res_object);
		$res_array = json_decode($res_json,TRUE);
		return($res_array);
	}
	
	
	//APIからデータ取得
	private function file_get_by_curl($url){
		$option = array(
			CURLOPT_RETURNTRANSFER	=> true, //文字列として返す
			CURLOPT_TIMEOUT			=> 3, // タイムアウト時間
		);
		
		//セッション初期化
		$ch = curl_init($url);
		curl_setopt_array($ch, $option);
		$res_data = curl_exec($ch);
		$res_info = curl_getinfo($ch);
		$res_errorNo = curl_errno($ch);
		$res_errorinfo = curl_error($ch);
		curl_close($ch);
		
		// エラー表示
		if ($res_errorNo !== 0) {
			if($this->debug_mode){
				echo 'curl実行エラー' . "\n";
				echo 'error no : ' . $res_errorNo . "\n";
				echo 'エラー内容 : ' . $res_errorinfo;
			}
			exit();
		}
		
		if ($res_info['http_code'] !== 200 && $res_info['http_code'] !== 500 && $res_info['http_code'] !== 503) {
			if($this->debug_mode){
				echo 'サーバーエラー　httpステータスコード : ' . $res_info['http_code'] . "\n";
				echo 'リクエストURL : ' . $res_info['url'] . "\n";
			}
			exit();
		}
		
		$this->curl_http_error_code = $res_info['http_code'];
		return $res_data;
	}
	
	
	//APIからのエラー表示
	private function display_API_error(){
		if($this->response['Items']['Request']['IsValid'] == 'False' or isset($this->response['Items']['Request']['Errors'])){
			echo 'APIからのエラー' . "\n";
			echo 'エラーコード : ' . $this->response['Items']['Request']['Errors']['Error']['Code'] . "\n" .
			'エラーメッセージ : ' . $this->response['Items']['Request']['Errors']['Error']['Message'] . "\n";
//			exit();
		}
	}
	
	
	//PEAR Cache_Lite チェック
	private function check_cache_lite(){
		if(isset($GLOBALS['cacheOptions'])){
			$this->cache_options = array_merge($this->cache_options, $GLOBALS['cacheOptions']);
		}
		
		if(!stream_resolve_include_path("Cache/Lite.php")){
			$this->check_cache_lite = false;
			return;
		}
		
		if(defined('CACHE_MODE')){
			if(CACHE_MODE === 'off'){
				$this->check_cache_lite = false;
				return;
			}
		}
				
		if(!empty($this->cache_options['cacheDir'])){
			$this->cache_options['cacheDir'] = 'cache/';
		}
		
		$cacheDir = $this->cache_options['cacheDir'];
		//キャッシュディレクトリがないとき、新規作成
		if(!file_exists($cacheDir)){
			mkdir($cacheDir, 705);
		}
		
	}
	
	
	//PHPバージョンチェック
	private function check_PHP_version(){
		$required_PHPversion = '5.3.0';
		if (version_compare(phpversion(), $required_PHPversion, '<')) {
			echo 'PHP' . $required_PHPversion . '以上が必須です。現在の環境は ' . phpversion() . ' です。' . "\n";
			exit();
		}
	}
	
	
	//デバッグモードチェック
	private function check_debug_mode(){
		if(defined('DEBUG_MODE')){
			if(DEBUG_MODE === 'on'){
				$this->debug_mode = true;
				ini_set('display_errors', '1');
				echo '<pre>';
			}
		}
	}
		
}
?>